import os
import sys
import json
import zipfile
import shutil
import xbmc
import xbmcgui
import xbmcplugin
import xbmcaddon
import urllib.request
import urllib.parse
import urllib.error
import ssl
from xbmcvfs import translatePath
from pathlib import Path


# Инициализация аддона
addon = xbmcaddon.Addon()
addon_id = addon.getAddonInfo('id')
addon_name = addon.getAddonInfo('name')
addon_path = addon.getAddonInfo('path')
addon_profile = translatePath(addon.getAddonInfo('profile'))

quality = addon.getSetting('quality')
quality_dict = {'Высокое': 'hi', 'Среднее': 'med', 'Низкое': 'low'}
quality = quality_dict[quality]

# URL для скачивания архива (можно изменить в настройках)
ARCHIVE_URL = 'http://stream.pcradio.ru/list/list_ru/list_old_ru.zip'
# Пароль архива (встроенный в код, не хранится в настройках)
ARCHIVE_PASSWORD = '78951233215987'  # Замените на ваш пароль

# Пути к файлам
JSON_FILENAME = 'list.json'
JSON_FILE_PATH = os.path.join(addon_profile, JSON_FILENAME)
TEMP_ARCHIVE_PATH = os.path.join(addon_profile, 'temp_data.zip')

# Создаем папку профиля, если её нет
if not os.path.exists(addon_profile):
    os.makedirs(addon_profile)

def log(message, level=xbmc.LOGINFO):
    """Логирование сообщений"""
    xbmc.log(f"[{addon_id}] {message}", level)

def download_and_extract_archive():
    """Скачивание и распаковка архива"""
    try:
        log("Начинаем скачивание архива...")
        
        # Скачиваем архив
        context = ssl._create_unverified_context() if hasattr(ssl, '_create_unverified_context') else None
        req = urllib.request.Request(ARCHIVE_URL)
        
        if context:
            response = urllib.request.urlopen(req, context=context)
        else:
            response = urllib.request.urlopen(req)
        
        with open(TEMP_ARCHIVE_PATH, 'wb') as f:
            f.write(response.read())
        
        log(f"Архив скачан: {TEMP_ARCHIVE_PATH}")
        
        # Распаковываем архив с паролем
        with zipfile.ZipFile(TEMP_ARCHIVE_PATH, 'r') as zip_ref:
            # Пытаемся извлечь с паролем
            try:
                zip_ref.extractall(addon_profile, pwd=ARCHIVE_PASSWORD.encode('utf-8'))
                log("Архив успешно распакован")
            except RuntimeError as e:
                log(f"Ошибка при распаковке с паролем: {str(e)}", xbmc.LOGERROR)
                # Пробуем без пароля (для обратной совместимости)
                try:
                    zip_ref.extractall(addon_profile)
                    log("Архив распакован без пароля")
                except Exception as e2:
                    log(f"Ошибка при распаковке без пароля: {str(e2)}", xbmc.LOGERROR)
                    raise Exception("Не удалось распаковать архив. Неверный пароль или поврежденный архив.")
        
        # Удаляем временный архив
        if os.path.exists(TEMP_ARCHIVE_PATH):
            os.remove(TEMP_ARCHIVE_PATH)
        
        # Проверяем, что JSON файл создан
        if not os.path.exists(JSON_FILE_PATH):
            # Ищем JSON файл в распакованных файлах
            json_found = False
            for root, dirs, files in os.walk(addon_profile):
                for file in files:
                    if file.endswith('.json'):
                        json_path = os.path.join(root, file)
                        shutil.move(json_path, JSON_FILE_PATH)
                        json_found = True
                        log(f"Найден и перемещен JSON файл: {file}")
                        break
                if json_found:
                    break
            
            if not json_found:
                raise Exception("В архиве не найден JSON файл")
        
        return True
        
    except Exception as e:
        log(f"Ошибка при скачивании/распаковке: {str(e)}", xbmc.LOGERROR)
        xbmcgui.Dialog().notification(
            addon_name,
            'Ошибка при обновлении данных',
            xbmcgui.NOTIFICATION_ERROR,
            5000
        )
        return False

def load_radio_data():
    """Загрузка данных о радиостанциях из JSON файла"""
    if not os.path.exists(JSON_FILE_PATH):
        log("JSON файл не найден, пытаемся скачать...")
        if not download_and_extract_archive():
            return None
    
    try:
        with open(JSON_FILE_PATH, 'r', encoding='utf-8') as f:
            data = json.load(f)
        log(f"Данные загружены из {JSON_FILE_PATH}")
        return data
    except Exception as e:
        log(f"Ошибка при чтении JSON: {str(e)}", xbmc.LOGERROR)
        return None

def show_main_menu():
    """Показать главное меню"""
    # Страны
    list_item = xbmcgui.ListItem(label='[COLOR blue]По странам[/COLOR]')
    list_item.setArt({'icon': 'DefaultCountry.png'})
    xbmcplugin.addDirectoryItem(
        handle=int(sys.argv[1]),
        url=f'plugin://{addon_id}/?action=countries',
        listitem=list_item,
        isFolder=True
    )
    
    # Жанры
    list_item = xbmcgui.ListItem(label='[COLOR green]По жанрам[/COLOR]')
    list_item.setArt({'icon': 'DefaultGenre.png'})
    xbmcplugin.addDirectoryItem(
        handle=int(sys.argv[1]),
        url=f'plugin://{addon_id}/?action=genres',
        listitem=list_item,
        isFolder=True
    )
    
    # Города
    list_item = xbmcgui.ListItem(label='[COLOR orange]По городам[/COLOR]')
    list_item.setArt({'icon': 'DefaultCity.png'})
    xbmcplugin.addDirectoryItem(
        handle=int(sys.argv[1]),
        url=f'plugin://{addon_id}/?action=cities',
        listitem=list_item,
        isFolder=True
    )
    
    # Все станции
    list_item = xbmcgui.ListItem(label='[COLOR yellow]Все станции[/COLOR]')
    list_item.setArt({'icon': 'DefaultMusic.png'})
    xbmcplugin.addDirectoryItem(
        handle=int(sys.argv[1]),
        url=f'plugin://{addon_id}/?action=all_stations',
        listitem=list_item,
        isFolder=True
    )
    
    # Поиск
    list_item = xbmcgui.ListItem(label='[COLOR red]Поиск станций[/COLOR]')
    list_item.setArt({'icon': 'DefaultSearch.png'})
    xbmcplugin.addDirectoryItem(
        handle=int(sys.argv[1]),
        url=f'plugin://{addon_id}/?action=search',
        listitem=list_item,
        isFolder=True
    )
    
    # Обновить данные
    list_item = xbmcgui.ListItem(label='[COLOR lightblue]Обновить данные с сервера[/COLOR]')
    list_item.setArt({'icon': 'DefaultAddonUpdate.png'})
    xbmcplugin.addDirectoryItem(
        handle=int(sys.argv[1]),
        url=f'plugin://{addon_id}/?action=update',
        listitem=list_item,
        isFolder=False
    )
    
    xbmcplugin.endOfDirectory(int(sys.argv[1]))

def show_countries():
    """Показать список стран"""
    data = load_radio_data()
    if not data:
        return
     
    for country in data['countries']:
        list_item = xbmcgui.ListItem(label=country['name'])
        list_item.setArt({'icon': 'DefaultCountry.png'})
        xbmcplugin.addDirectoryItem(
            handle=int(sys.argv[1]),
            url=f'plugin://{addon_id}/?action=country_stations&country={country["id"]}',
            listitem=list_item,
            isFolder=True
        )
    
    xbmcplugin.endOfDirectory(int(sys.argv[1]))


def show_genres():
    """Показать список жанров"""
    data = load_radio_data()
    if not data:
        return

    for genre in data['genres']:
        list_item = xbmcgui.ListItem(label=genre['name'])
        list_item.setArt({'icon': 'DefaultGenre.png'})
        xbmcplugin.addDirectoryItem(
            handle=int(sys.argv[1]),
            url=f'plugin://{addon_id}/?action=genre_stations&genre={genre["id"]}',
            listitem=list_item,
            isFolder=True
        )
    
    xbmcplugin.endOfDirectory(int(sys.argv[1]))


def show_cities():
    """Показать список городов"""
    data = load_radio_data()
    if not data:
        return

    for city in data['cities']:
        list_item = xbmcgui.ListItem(label=city['name'])
        list_item.setArt({'icon': 'DefaultCity.png'})
        xbmcplugin.addDirectoryItem(
            handle=int(sys.argv[1]),
            url=f'plugin://{addon_id}/?action=city_stations&city={city["id"]}',
            listitem=list_item,
            isFolder=True
        )
    
    xbmcplugin.endOfDirectory(int(sys.argv[1]))

def show_all_stations():
    """Показать все станции"""
    data = load_radio_data()
    if not data:
        return
    
    stations = data.get('stations', [])
    show_stations_list(stations, "Все станции")

def show_country_stations(country):
    """Показать станции по стране"""
    data = load_radio_data()
    if not data:
        return
    
    country = urllib.parse.unquote(country)
    stations = [s for s in data.get('stations', []) if s.get('country_id') == int(country)]
    show_stations_list(stations, f"Страна: {country}")

def show_genre_stations(genre):
    """Показать станции по жанру"""
    data = load_radio_data()
    if not data:
        return
    
    genre = urllib.parse.unquote(genre)
    stations = []
    for station in data.get('stations', []):
        station_genres = station.get('genres_ids', [])
        if isinstance(station_genres, str):
            station_genres = [station_genres]
        
        if int(genre) in station_genres:
            stations.append(station)
    
    show_stations_list(stations, f"Жанр: {genre}")

def show_city_stations(city):
    """Показать станции по городу"""
    data = load_radio_data()
    if not data:
        return

    city = urllib.parse.unquote(city)
    stations = []
    for station in data.get('stations', []):
        station_cities = station.get('cities_ids', [])
        if isinstance(station_cities, str):
            station_cities = [station_cities]
        
        if int(city) in station_cities:
            stations.append(station)
    
    show_stations_list(stations, f"Город: {city}")

def show_stations_list(stations, title=""):
    """Показать список станций"""
    if not stations:
        xbmcgui.Dialog().ok(addon_name, 'Станции не найдены')
        return
    
    # Сортируем станции по названию
    stations.sort(key=lambda x: x.get('name', '').lower())
    
    for station in stations:
        name = station.get('name', 'Без названия')
        url = station.get('stream', '')+'-'+quality
        icon = station.get('logo', '')
        descr = station.get('descr', '')
        
        # Создаем элемент списка
        list_item = xbmcgui.ListItem(label=name)
        
        if icon:
            list_item.setArt({'icon': icon, 'thumb': icon})

        tags = list_item.getMusicInfoTag()
        tags.setTitle(name)
        tags.setComment(descr)
        
        # Устанавливаем свойство "IsPlayable"
        list_item.setProperty('IsPlayable', 'true')
        
        # Добавляем элемент в список плагина
        xbmcplugin.addDirectoryItem(
            handle=int(sys.argv[1]),
            url=url,
            listitem=list_item,
            isFolder=False
        )
    
    xbmcplugin.endOfDirectory(int(sys.argv[1]))

def search_stations():
    """Поиск станций по названию"""
    keyboard = xbmc.Keyboard('', 'Поиск радиостанций')
    keyboard.doModal()
    
    if keyboard.isConfirmed():
        search_text = keyboard.getText().lower()
        if search_text:
            data = load_radio_data()
            if not data:
                return
            
            # Ищем станции по названию
            found_stations = []
            for station in data.get('stations', []):
                name = station.get('name', '').lower()
                if search_text in name:
                    found_stations.append(station)
            
            if found_stations:
                show_stations_list(found_stations, f"Результаты поиска: {search_text}")
            else:
                xbmcgui.Dialog().ok(addon_name, f'Станции по запросу "{search_text}" не найдены')

def play_radio(url):
    """Воспроизведение радиостанции"""
    list_item = xbmcgui.ListItem(path=url)
    list_item.setProperty('IsPlayable', 'true')
    xbmcplugin.setResolvedUrl(int(sys.argv[1]), True, list_item)

def update_data():
    """Обновление данных с сервера"""
    progress_dialog = xbmcgui.DialogProgress()
    progress_dialog.create(addon_name, 'Обновление данных...')
    
    progress_dialog.update(0, 'Скачивание архива...')
    if download_and_extract_archive():
        progress_dialog.update(100, 'Данные успешно обновлены!')
        xbmc.sleep(1000)
        xbmcgui.Dialog().notification(
            addon_name,
            'Данные успешно обновлены',
            xbmcgui.NOTIFICATION_INFO,
            3000
        )
    else:
        progress_dialog.update(100, 'Ошибка при обновлении данных')
        xbmc.sleep(1000)

    progress_dialog.close()
    
    # Обновляем список
    xbmc.executebuiltin('Container.Refresh()')

def clear_cache():
    """Очистка кэша"""
    try:
        # Удаляем JSON файл
        if os.path.exists(JSON_FILE_PATH):
            os.remove(JSON_FILE_PATH)
            log("Кэш очищен")
            xbmcgui.Dialog().notification(
                addon_name,
                'Кэш очищен',
                xbmcgui.NOTIFICATION_INFO,
                3000
            )
        else:
            xbmcgui.Dialog().notification(
                addon_name,
                'Кэш уже пуст',
                xbmcgui.NOTIFICATION_INFO,
                3000
            )
    except Exception as e:
        log(f"Ошибка при очистке кэша: {str(e)}", xbmc.LOGERROR)
        xbmcgui.Dialog().notification(
            addon_name,
            'Ошибка при очистке кэша',
            xbmcgui.NOTIFICATION_ERROR,
            3000
        )

def router(paramstring):
    """Маршрутизатор команд"""
    params = dict(urllib.parse.parse_qsl(paramstring))
    
    action = params.get('action', '')
    country = params.get('country', '')
    genre = params.get('genre', '')
    city = params.get('city', '')
    url = params.get('url', '')
    
    if action == 'update':
        update_data()
    elif action == 'clear_cache':
        clear_cache()
    elif action == 'countries':
        show_countries()
    elif action == 'genres':
        show_genres()
    elif action == 'cities':
        show_cities()
    elif action == 'all_stations':
        show_all_stations()
    elif action == 'country_stations' and country:
        show_country_stations(country)
    elif action == 'genre_stations' and genre:
        show_genre_stations(genre)
    elif action == 'city_stations' and city:
        show_city_stations(city)
    elif action == 'search':
        search_stations()
    elif url:
        play_radio(url)
    else:
        show_main_menu()

if __name__ == '__main__':
    router(sys.argv[2][1:])
    