import math
import xbmc
import textwrap
import hashlib
import uuid
import time
import json
from functools import wraps
from itertools import chain
import resources.lib.utils as utils
from resources.lib.utils import show_progress, cache
from resources.lib.translation import _

from urllib.parse import urlencode
import urllib3
urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)

CACHE_MAX_AGE = 1800  # 30 minutes
#ANDROID_HOST = 'https://discovery-android-26.ertelecom.ru'
ANDROID_HOST = utils.addon.getSettingString('android_host')
#STB_HOST = 'https://discovery-stb3.ertelecom.ru'
STB_HOST = utils.addon.getSettingString('stb_host')
DEVICE_ID = 'kodi:' + hashlib.sha1(str(uuid.getnode()).encode('utf8')).hexdigest()[-16:]

HEADERS = {
    'View': 'stb3',
    'X-Device-Info': DEVICE_ID,
    'X-App-Version': '3.12.0',
    'User-Agent': xbmc.getUserAgent()
}


class ApiRequestError(Exception):
    def __init__(self, message):
        super(ApiRequestError, self).__init__(message)
        self.error = {'message': message}


class ApiResponseError(Exception):
    def __init__(self, error):
        super(ApiResponseError, self).__init__(error.get('message'))
        self.error = error


def _headers(token):
    headers = dict(HEADERS)
    headers.update({'X-Auth-Token': token})
    return headers

def _headers_urlencoded(token):
    headers = _headers(token)
    headers['Content-Type'] = 'application/x-www-form-urlencoded'
    return headers

@show_progress(_('progress.device_registration'))
def token():
    """Requests initial token is_bound=false"""
    url = ANDROID_HOST + '/token/device'
    sid = utils.read_file('device_id.txt', '').strip()
    if sid == '':
       sid = DEVICE_ID
    params = dict(
        client_id='er_android_device',
        device_id=sid,
        timestamp=str(int(time.time()))
    )
    resp = _request('get', url, fields=params, headers=HEADERS)
    #utils.write_file('00_resp_token.txt', resp)
    return utils.subset(resp, 'is_bound', 'token', 'expires')


def token_refresh(token):
    url = ANDROID_HOST + '/token/refresh'
    resp = _request('get', url, headers=_headers(token))
    return utils.subset(resp, 'token', 'expires')


@cache(CACHE_MAX_AGE)
@show_progress(_('progress.regions'))
def regions(token):
    url = ANDROID_HOST + '/er/misc/domains'
    resp = _request('get', url, headers=_headers(token))
    domains = []
    for item in resp['domains']:
        if item['code']:
            domain = utils.subset(item, 'extid', 'code', 'title')
            domains.append(domain)
    return domains


@show_progress(_('progress.auth'))
def auth(token, username, password, region):
    """Requests sso and then requests token is_bound=true"""

    url = ANDROID_HOST + '/er/ssoauth/auth'
    data = {'username': username, 'password': password, 'region': region}
    resp = _request('post', url, headers=_headers_urlencoded(token), body=urlencode(data))

    url = ANDROID_HOST + '/token/subscriber_device/by_sso'
    params = dict(sso_system='er', sso_key=resp['sso'])
    resp = _request('get', url, fields=params, headers=_headers(token))
    return utils.subset(resp, 'is_bound', 'token', 'expires')


@show_progress(_('progress.sms_request'))
def sms_auth(token, phone):
    url = ANDROID_HOST + '/er/ott/get_agreements_by_phone'
    params = {'phone_number': phone}
    resp = _request('get', url, fields=params, headers=_headers(token))
    if not resp['principals']:
        raise ApiResponseError({'message': _('error.contract_not_found') % phone})

    url = ANDROID_HOST + '/er/sms/auth'
    region = resp['principals'][0]['domain']
    data = {'phone': phone, 'region': region}
    resp = _request('post', url, headers=_headers_urlencoded(token), body=urlencode(data))
    result = utils.subset(
        resp['agreements'],
        'send_sms',
        ('sms_error_text', 'message'),
        (lambda d: d['agreement'][0]['agr_id'], 'agr_id')
    )
    result['region'] = region
    return result


@show_progress(_('progress.sms_check'))
def sms_check(token, phone, region, agr_id, sms_code):
    url = ANDROID_HOST + '/er/sms/check'
    data = {'phone': phone, 'region': region, 'agr_id': str(agr_id), 'sms_code': str(sms_code)}
    resp = _request('post', url, headers=_headers_urlencoded(token), body=urlencode(data))
    if not resp.get('token'):
        raise ApiResponseError({'message': resp['Agreements']['sms_error_text']})
    return utils.subset(resp, 'is_bound', 'token', 'expires')


@cache(7200)
@show_progress(_('progress.status'))
def status(token):
    url = ANDROID_HOST + '/er/multiscreen/status'
    resp = _request('get', url, headers=_headers(token))
    return resp['status']


@show_progress(_('progress.binding'))
def bind_device(token):
    url = ANDROID_HOST + '/er/multiscreen/device/bind'
    device_title = utils.addon.getSetting('device_title').strip()
    if device_title == '':
        device_title = 'Kodi'
    _request('post', url, headers=_headers_urlencoded(token), body=urlencode({'title': device_title}))


# Used only in tests
def devices(token):
    url = ANDROID_HOST + '/er/multiscreen/devices'
    resp = _request('get', url, headers=_headers(token))
    return resp['devices']


def unbind_device(token, id):
    url = ANDROID_HOST + '/er/multiscreen/device/unbind'
    _request('post', url, headers=_headers_urlencoded(token), body=urlencode({'device_id': id}))


@cache(CACHE_MAX_AGE)
@show_progress(_('text.channels'))
def channels(token, limit, page):
    url = STB_HOST + '/api/v3/showcases/library/channels'
    params = {'limit': str(limit), 'page': str(page)}
    resp = _request('get', url, fields=params, headers=_headers(token))
    #utils.write_file('00_resp_ch_' + str(page) + '.txt', json.dumps(resp, indent=4).encode('utf-8').decode('unicode-escape'))
    chs = _map_items(resp['data']['items'], ['id', 'title', 'description', 'lcn'], {
        'hls_id': ('hls_mcast_local', 'hls'),
        'poster_id': 'poster_channel_grid_blueprint'
    })
    return {'channels': chs, 'pages': _pages(resp, limit)}


@cache(CACHE_MAX_AGE)
@show_progress(_('text.channels'))
def channels_all(token, public=False):
    url = ANDROID_HOST + '/channel_list/multiscreen'
    resp = _request('get', url, headers=_headers(token))
    #utils.write_file('00_resp_ch_all.txt', json.dumps(resp, indent=4).encode('utf-8').decode('unicode-escape'))
    chs = []
    for item in resp['collection']:
        ch = utils.subset(item, 'id', 'title', 'description', ('er_lcn', 'lcn'))
        resources = {r['category']: r for r in item['resources']}
        if public and not resources.get('hls', {}).get('is_public'):
            continue
        hls_id = resources.get('hls', {}).get('id')
        if hls_id == None:
            hls_id = resources.get('hls_mcast_local', {}).get('id')
        res = {
            'hls_id': hls_id,
            'poster_id': resources.get('poster_channel_grid_blueprint', {}).get('id')
        }
        ch.update(res)
        chs.append(ch)
    chs = sorted(chs, key=lambda x: x['lcn'])
    return {'channels': chs}


@show_progress(_('progress.epg'))
def epg_get_program_list(token, date_from, date_to, channel):
    url = STB_HOST + '/epg/get_schedule'
    params = {'select': 'channel_id,id,start,end,duration,title,description', 'channel_id': str(channel), 'start_from': str(date_from), 'star_to': str(date_to)}
    return _request('get', url, fields=params, headers=_headers(token))
   
     
def playlist_url(token, channel_id, hls_id):
    url = ANDROID_HOST + '/resource/get_url/%i/%i' % (channel_id, hls_id)
    resp = _request('get', url, headers=_headers(token))
    return resp['url']


def playlist_url_hq(token, channel_id, hls_id):
    url = ANDROID_HOST + '/resource/get_url/%i/%i' % (channel_id, hls_id)
    resp = _request('get', url, headers=_headers(token))
    playlist_fix_url_hq(resp)
    return resp['url']


def playlist_fix_url_hq(responce):
    text = _download_text(responce['url'])
    url = extract_stream_url(text)
    if url != None:
        responce['url'] = url


def extract_stream_url(text):
    url = None
    #utils.write_file('a9_m38_list.txt', text)
    if text.startswith('#EXTM3U'):
        lst_lines = text.splitlines()
        for line in lst_lines:
            if not line.startswith('#EXT') and line.find('track.m3u8')>0:
                url = line
#    if url != None:
#        ndx = url.find('track_2')
#        if ndx>0:
#            url = url.replace('track_2', 'track_3')
#        else:
#            url = url.replace('track_1', 'track_2')
    return url


def art_url(art_id, w=0, h=0):
    url = 'https://er-cdn.ertelecom.ru/content/public/r'
    if isinstance(art_id, int):
        url = 'https://er-cdn.ertelecom.ru/content/public/r%i' % art_id
    if w and h:
        url += '/%ix%i:crop' % (w, h)
    return url


@cache(CACHE_MAX_AGE)
@show_progress(_('text.channel_packages'))
def channel_packages(token, limit, page):
    url = STB_HOST + '/api/v3/showcases/library/channel-packages'
    params = {'limit': str(limit), 'page': str(page)}
    resp = _request('get', url, fields=params, headers=_headers(token))
    pkgs = _map_items(resp['data']['items'],
                      ['id', 'title', 'description', (lambda i: i['adult']['type'], 'adult')], {
        'poster_id': '3_smarttv_package_poster_video_library_blueprint',
        'fanart_id': '3_smarttv_asset_background_banner_fullscreen_blueprint'
    })
    return {'packages': pkgs, 'pages': _pages(resp, limit)}


@cache(CACHE_MAX_AGE)
@show_progress(_('progress.package_channels'))
def package_channels(token, id, adult=0):
    url = STB_HOST + '/api/v3/showcases/children/channel-package/%i/channels' % id
    params = {'adult': 'adult,not-adult'} if adult else None
    resp = _request('get', url, fields=params, headers=_headers(token))
    chs = _map_items(resp['data']['items'], ['id', 'title', 'description', 'lcn'], {
        'hls_id': 'hls',
        'poster_id': 'poster_channel_grid_blueprint'
    })
    return {'channels': chs}


@cache(CACHE_MAX_AGE)
@show_progress(_('text.movies'))
def movies(token, limit, offset, free):
    url = STB_HOST + '/api/v3/showcases/library/' + ('freemovies' if free else 'movies')
    params = {'limit': str(limit), 'offset': str(offset)}
    resp = _request('get', url, fields=params, headers=_headers(token))
    #utils.write_file('m0_resp_movies.txt', json.dumps(resp, indent=4).encode('utf-8').decode('unicode-escape'))
    movs = _map_items(resp['data']['items'], ['id', 'title', 'description'], {
        'hls_id': 'hls',
        'poster_id': 'poster_blueprint',
        'fanart_id': '3_smarttv_asset_background_video_library_blueprint'
    })
    return {'movies': movs, 'offset': offset + limit, 'total': resp['data']['total']}


@cache(CACHE_MAX_AGE)
@show_progress(_('text.serials'))
def serials(token, limit, offset):
    params = {'limit': str(limit), 'offset': str(offset)}
    url = STB_HOST + '/api/v3/showcases/library/serials'
    resp = _request('get', url, fields=params, headers=_headers(token))
    srls = _map_items(resp['data']['items'], ['id', 'title', 'description'], {
        'hls_id': 'hls',
        'poster_id': 'poster_blueprint',
        'fanart_id': '3_smarttv_serial_background_video_library_blueprint'
    })
    return {'serials': srls, 'offset': offset + limit, 'total': resp['data']['total']}


def _filter_available(items):
    length = len(items)
    fl = filter(lambda i: i['available']['type'] != 'not-available', items)
    return length, fl


@cache(86400)  # 1 day
@show_progress(_('text.available_serials'))
def serials_available(token, limit, offset, set_progress):
    params = {'limit': '100', 'offset': str(offset)}
    url = STB_HOST + '/api/v3/showcases/library/serials'
    headers = _headers(token)
    resp = _request('get', url, fields=params, headers=headers)
    items_total = resp['data']['total']

    length, filtered = _filter_available(resp['data']['items'])
    items = list(filtered)
    set_progress(len(items) / limit * 100)
    offset += length

    while len(items) < limit and offset < items_total:
        params.update({'offset': str(offset)})
        resp = _request('get', url, fields=params, headers=headers)
        length, filtered = _filter_available(resp['data']['items'])
        items = list(chain(items, filtered))
        set_progress(len(items) / limit * 100)
        offset += length

    srls = _map_items(items, ['id', 'title', 'description'], {
        'hls_id': 'hls',
        'poster_id': 'poster_blueprint',
        'fanart_id': '3_smarttv_serial_background_video_library_blueprint'
    })
    return {'serials': srls, 'offset': offset, 'total': items_total}


@cache(CACHE_MAX_AGE)
@show_progress(_('text.seasons'))
def seasons(token, serial_id):
    url = STB_HOST + '/api/v3/showcases/seasons/serial/%i/seasons' % serial_id
    resp = _request('get', url, headers=_headers(token))
    sns = _map_items(resp['data']['items'], ['id', 'title', 'description', 'number'], {
        'poster_id': 'poster_blueprint',
        'fanart_id': '3_smarttv_season_background_video_library_blueprint'
    })
    return {'seasons': sns}


@cache(CACHE_MAX_AGE)
@show_progress(_('text.episodes'))
def episodes(token, season_id):
    url = STB_HOST + '/api/v3/showcases/episodes/season/%i/episodes' % season_id
    resp = _request('get', url, headers=_headers(token))
    #utils.write_file('s0_resp_serials.txt', json.dumps(resp, indent=4).encode('utf-8').decode('unicode-escape'))
    epds = _map_items(resp['data']['items'], ['id', 'title', 'description', 'number'], {
        'hls_id': 'hls',
        'poster_id': [
            '3_smarttv_episode_poster_video_library_blueprint',
            '3_smarttv_tvshow_poster_video_library_blueprint',
            'poster_blueprint'
        ]
    })
    return {'episodes': epds}


def _map_items(items, keys, res_map):
    mapped = []
    for item in items:
        mi = utils.subset(item, *keys)
        resources = {r['type']: r for r in item['resources']}
        res = {'available': item['available']['type'] != 'not-available'}
        for data_key, res_type in res_map.items():
            if isinstance(res_type, str):
                res[data_key] = resources.get(res_type, {}).get('id')
            else:
                res[data_key] = next((resources[t]['id'] for t in res_type if resources.get(t)), None)
        mi.update(res)
        mapped.append(mi)
    return mapped


def _request(method, url, **kwargs):
    resp = None
    try:
        http = urllib3.PoolManager(cert_reqs='CERT_NONE')
        resp = http.request(method, url, **kwargs) 
        if utils.addon.getSettingBool('debug_logging'):
           utils.log('=========================================================================================')
           utils.log('URL     : ' + url)
           if 'fields' in kwargs: 
               utils.log('FIELDS  : ' + str(kwargs.get('fields')))
           if 'body' in kwargs: 
               utils.log('BODY    : ' + str(kwargs.get('body')))
           utils.log('RESPONSE: ' + (resp.data.decode('utf-8')).encode('utf-8').decode('unicode-escape'))
           utils.log('=========================================================================================')
    except Exception as ex:
        template = "An exception of type {0} occurred. Arguments:\n{1!r}"
        msg = template.format(type(ex).__name__, ex.args)
    if resp == None:
        raise ApiRequestError(msg)

    try:
        data = resp.json()
    except Exception as ex:
        utils.log('Failed to retrieve json structure.\n')
        raise ApiResponseError({'message': 'Failed to retrieve json structure'})

    if 'error' in data:
        raise ApiResponseError(data['error'])
    return data


def _download_text(url):
    resp = None
    try:
        http = urllib3.PoolManager(cert_reqs='CERT_NONE')
        resp = http.request('get', url)
    except Exception as ex:
        template = "An exception of type {0} occurred. Arguments:\n{1!r}"
        msg = template.format(type(ex).__name__, ex.args)
    if resp == None:
        raise ApiRequestError(msg)

    return resp.data.decode('utf-8')


def _pages(resp, limit):
    return int(math.ceil(resp['data']['total'] / float(limit)))


def on_error(callback):
    def decorator(func):

        @wraps(func)
        def wrapper(router, params):
            try:
                func(router, params)
            except (ApiRequestError, ApiResponseError) as e:
                utils.show_error(e.error)
                args = [router] if callback.func_code.co_argcount == 1 else [router, params]
                callback(*args)

        return wrapper
    return decorator
