"""IPTV Manager Integration module"""

import xbmcplugin
import xbmcgui
import resources.lib.api as api
import resources.lib.utils as utils

import json
import socket

CHANNEL_IDS = 'channel_ids.list'

class IPTVManager:
    """Interface to IPTV Manager"""

    def __init__(self, port):
        """Initialize IPTV Manager object"""
        self.port = port

    def via_socket(func):
        """Send the output of the wrapped function to socket"""

        def send(self, router):
            """Decorator to send over a socket"""
            sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
            sock.connect(('127.0.0.1', self.port))
            try:
                sock.sendall(json.dumps(func(self, router)).encode())
            finally:
                sock.close()

        return send

    @via_socket
    def send_channels(self, router):
        """Return JSON-STREAMS formatted information to IPTV Manager"""

        utils.log('Loading Channels ...')

        handle = router.session.handle
        token = router.session.token

        if not token:
             token = api.token()
             utils.write_file('token.json', token)
             utils.update_token_info(token['token'])

        exc_ch = utils.addon.getSetting('channels_exclude')
        exc_ch_list = utils.expand_list(exc_ch)
        #utils.write_file('a1_exc_ch_list.txt', exc_ch_list)

        resp = api.channels_all(token['token'], not token['is_bound'])
        #utils.write_file('a2_resp_multicast.txt', resp)

        ch_ids = []
        channels = []
        for ch in resp['channels']:
             if ch['lcn'] not in exc_ch_list:
                 channels.append(dict(
                    id='movix-{id}'.format(id=ch['id']),
                    name=ch['title'],
                    logo=api.art_url(ch['poster_id']),
                    stream=router.root_url('play', id=ch['id'], hls_id=ch['hls_id']),
                    preset=ch['lcn'],
                 ))
                 ch_ids.append(ch['id'])
        #utils.write_file('a2_channels_multicast.txt', channels)

        utils.write_file(CHANNEL_IDS, ch_ids)
        return dict(version=1, streams=channels)

    @via_socket
    def send_epg(self, router):
        """Return JSON-EPG formatted information to IPTV Manager"""

        utils.log('Loading EPG ...')

        handle = router.session.handle
        token = router.session.token

        if not token:
             token = api.token()
             utils.write_file('token.json', token)
             utils.update_token_info(token['token'])

        epg_data = dict()
        epg_map = json.loads(utils.read_file(CHANNEL_IDS, '[]'))
        if len(epg_map) > 0:

            from datetime import datetime, timedelta
            date = datetime.now()
            tzs = date.astimezone().strftime('%z')
            if tzs[0:1] == '-':
                multiplier = -1
            else:
                multiplier = 1
            tz = timedelta(hours=(int(tzs[1:3]) * multiplier), minutes=(int(tzs[3:5]) * multiplier))
            date_from = int((date - tz + timedelta(hours=-4)).timestamp())
            date_to = int((date - tz + timedelta(days=1) + timedelta(hours=+8)).timestamp())

            keys = ','.join(list(map(str, epg_map)))
            epg_content = api.epg_get_program_list(token['token'], date_from, date_to, keys)
            #utils.write_file('b1_epg_content_' + str(i) + '.txt', epg_content)
            for item in epg_content['channels']:
                schedules = []
                for si in item['schedule']:
                    if type(si['start']) is not None \
                            and type(si['end']) is not None \
                            and type(si['title']) is not None:
                        schedules.append(dict(
                            start=(datetime.fromtimestamp(int(si['start']))).astimezone().strftime('%Y-%m-%dT%H:%M:%S%z'),
                            stop=(datetime.fromtimestamp(int(si['end']))).astimezone().strftime('%Y-%m-%dT%H:%M:%S%z'),
                            title=si['title'],
                            description=str(si['description']),
                            # todo: find host of icons for epg
                            # image=si['icon']
                        ))
                epg_data['movix-{id}'.format(id=item['id'])] = schedules

        #utils.write_file('b2_epg_data.txt', epg_data)
        return dict(version=1, epg=epg_data)

def channels(router, params):
    IPTVManager(params['port']).send_channels(router)

def epg(router, params):
    IPTVManager(params['port']).send_epg(router)

