def get_weather_code(ph_hint, fog, time_str, cloud_cover, wind_velocity_hint, sunrise, sunset):
    """
    Определяет код погоды по входным параметрам.
    
    Args:
        ph_hint (str): Описание осадков
        fog (int): Процент тумана (0-100)
        time_str (str): Время в формате "ЧЧ:ММ"
        cloud_cover (int): Процент облачности (0-100)
        wind_velocity_hint (int): Уровень ветра (0-12)
        sunrise (str): Время восхода "ЧЧ:ММ"
        sunset (str): Время заката "ЧЧ:ММ"
        
    Returns:
        int/str: Код погоды из списка [0-47, 'na']
    """
    
    # Определяем время суток по восходу и закату
    is_daytime = _is_daytime(time_str, sunrise, sunset)
    
    # Основная логика сопоставления по ph_hint (осадкам)
    if ph_hint == 'Без осадков':
        return _get_no_precipitation_code(fog, cloud_cover, wind_velocity_hint, is_daytime)
    elif ph_hint == 'Преимущественно без осадков':
        return _get_mostly_clear_code(cloud_cover, is_daytime)
    elif 'дождь' in ph_hint.lower() and 'снег' not in ph_hint.lower():
        return _get_rain_code(ph_hint, wind_velocity_hint)
    elif 'снег' in ph_hint.lower() and 'дождь' not in ph_hint.lower():
        return _get_snow_code(ph_hint, wind_velocity_hint)
    elif 'дождь со снегом' in ph_hint.lower() or 'снег с дождем' in ph_hint.lower():
        return _get_mixed_precipitation_code(ph_hint)
    elif 'переохлаждённый дождь' in ph_hint.lower():
        return _get_freezing_rain_code(ph_hint)
    
    return 'na'

def _is_daytime(time_str, sunrise, sunset):
    """Определяет, день ли сейчас по времени восхода и заката"""
    def time_to_minutes(t):
        h, m = map(int, t.split(':'))
        return h * 60 + m
    
    current_minutes = time_to_minutes(time_str)
    sunrise_minutes = time_to_minutes(sunrise)
    sunset_minutes = time_to_minutes(sunset)
    
    # Если закат позже восхода (нормальный день)
    if sunset_minutes > sunrise_minutes:
        return sunrise_minutes <= current_minutes < sunset_minutes
    else:
        # Если закат раньше восхода (переход через полночь)
        return current_minutes >= sunrise_minutes or current_minutes < sunset_minutes

def _get_no_precipitation_code(fog, cloud_cover, wind_level, is_daytime):
    """Коды для погоды без осадков"""
    
    # Туман и дымка (приоритет)
    if fog >= 80:
        return 20  # foggy
    elif fog >= 50:
        return 21  # haze
    elif fog >= 30:
        return 22  # smoky
    
    # Экстремальный ветер
    if wind_level >= 12:
        return 2   # hurricane
    elif wind_level >= 10:
        return 1   # tropical storm
    elif wind_level >= 8:
        return 23  # blustery (очень крепкий ветер)
    elif wind_level >= 6:
        return 24  # windy (сильный ветер)
    elif wind_level >= 4:
        return 24  # windy (умеренный/свежий ветер)
    
    # Облачность
    if cloud_cover >= 90:
        return 26  # cloudy
    elif cloud_cover >= 70:
        return 28 if is_daytime else 27  # mostly cloudy
    elif cloud_cover >= 40:
        return 30 if is_daytime else 29  # partly cloudy
    elif cloud_cover >= 20:
        return 34 if is_daytime else 33  # fair
    else:
        return 32 if is_daytime else 31  # clear/sunny

def _get_mostly_clear_code(cloud_cover, is_daytime):
    """Коды для преимущественно без осадков"""
    if cloud_cover >= 70:
        return 28 if is_daytime else 27  # mostly cloudy
    elif cloud_cover >= 40:
        return 30 if is_daytime else 29  # partly cloudy
    else:
        return 44  # partly cloudy (общий)

def _get_rain_code(ph_hint, wind_level):
    """Коды для дождя"""
    intensity = ph_hint.split()[0]  # Слабый/Умеренный/Сильный
    
    # Экстремальные условия с дождем
    if wind_level >= 12:
        return 0   # tornado
    elif wind_level >= 10:
        return 1   # tropical storm
    elif wind_level >= 8:
        return 2   # hurricane
    
    # Грозы (сильный ветер)
    if wind_level >= 5:  # Свежий ветер и выше
        if intensity == 'Сильный':
            return 3   # severe thunderstorms
        elif intensity == 'Умеренный':
            return 4   # thunderstorms
        else:
            return 37  # isolated thunderstorms
    
    # Ливни (умеренный ветер)
    if wind_level >= 4 and intensity == 'Сильный':
        return 11  # showers
    
    # Обычный дождь
    if intensity == 'Слабый':
        return 9   # drizzle
    elif intensity == 'Умеренный':
        return 12  # showers
    else:  # Сильный
        if wind_level >= 4:
            return 40  # scattered showers
        else:
            return 11  # showers

def _get_snow_code(ph_hint, wind_level):
    """Коды для снега"""
    intensity = ph_hint.split()[0]  # Слабый/Умеренный/Сильный
    
    # Снег с сильным ветром
    if wind_level >= 6 and intensity == 'Сильный':
        return 15  # blowing snow
    
    # Снежные ливни
    if wind_level >= 4 and intensity == 'Сильный':
        return 42  # scattered snow showers
    
    # Обычный снег
    if intensity == 'Слабый':
        if wind_level >= 2:
            return 13  # snow flurries
        else:
            return 14  # light snow showers
    elif intensity == 'Умеренный':
        return 16  # snow
    else:  # Сильный
        return 41  # heavy snow

def _get_mixed_precipitation_code(ph_hint):
    """Коды для смешанных осадков"""
    intensity = ph_hint.split()[0]
    
    if 'дождь со снегом' in ph_hint:
        if intensity == 'Слабый':
            return 5  # mixed rain and snow
        elif intensity == 'Умеренный':
            return 6  # mixed rain and sleet
        else:  # Сильный
            return 35  # mixed rain and hail
    else:  # снег с дождем
        return 7  # mixed snow and sleet

def _get_freezing_rain_code(ph_hint):
    """Коды для переохлажденного дождя"""
    intensity = ph_hint.split()[0]
    
    if intensity == 'Слабый':
        return 8   # freezing drizzle
    elif intensity == 'Умеренный':
        return 10  # freezing rain
    else:  # Сильный
        return 10  # freezing rain
    
