import requests
import xbmc
import xbmcgui
import xbmcaddon
import json
from datetime import datetime
from rp5 import search_city, get_weather, get_phenomenon, get_wind_direction
from conditions import get_weather_code


window = xbmcgui.Window(12600)
ADDON = xbmcaddon.Addon()


class LocationSearch:
    def __init__(self):
        self.dialog = xbmcgui.Dialog()
        self.keyboard = xbmc.Keyboard()
    
    def search_location(self):
        self.keyboard.setHeading("Введите название города")
        self.keyboard.doModal()        
        if self.keyboard.isConfirmed():
            query = self.keyboard.getText()
            if query:
                return self._perform_search(query)
        return None
    
    def _perform_search(self, query):
        locations = search_city(query)      
        if not locations:
            self.dialog.ok("Погода от rp5", "Локации не найдены") 
            return None
        return self._show_location_list(locations)
    
    
    def _show_location_list(self, locations):
        items = [f"{location['name']}" for location in locations]        
        selected = self.dialog.select("Выберите локацию", items)
        if selected >= 0:
            return locations[selected]
        return None


def handle_location_search(loc_id=1):
    search = LocationSearch()
    location = search.search_location()    
    if location:
        # Сохраняем выбранную локацию в настройках
        ADDON.setSetting(f"loc{loc_id}_id", str(location["id"]))
        ADDON.setSetting(f"loc{loc_id}_name", location["unique"])        
        # Показываем подтверждение
        xbmcgui.Dialog().ok("Погода от rp5", f"Местоположение установлено: {location['unique']}")
        return list(location)
    return []


def set_weather (current: dict, daily: list, hourly: list, args: str):
    # Заполняем шапку
    location_value = window.getProperty(f"Location{args}")
    window.setProperty("Current.Location", location_value)
    window.setProperty("WeatherProvider", "Погода от rp5")

    window.setProperty("Current.Temperature", f"{round(current['temperature']['c'])}°C")
    window.setProperty("Current.FeelsLike", f"{round(current['feel_temperature']['c'])}°C")
    
    ph_hint, rain_mm, snow_mm, fog = get_phenomenon(current['phenomenon'])
    cloud_cover = current['cloud_cover']['pct']
#    cloud_cover_hint = current['cloud_cover_hint']['pct'].split('\n')[0]
#    window.setProperty("Current.Condition", f"{ph_hint}, {cloud_cover_hint}")
    window.setProperty("Current.Condition", f"{ph_hint}")
    window.setProperty("Current.Precipitation", f"{rain_mm or snow_mm} мм")
    window.setProperty("Current.Cloudiness", f"{cloud_cover}")
    

    # В шапке этого значка не будет, но он появится под часами, если включить опцию
    # "Интерфейс - Настройки обложки - Показать сведения о погоде на верхней панели"
    sunrise = ru_format_date(current['sunrise'])[2]
    sunset = ru_format_date(current['sunset'])[2]
    icon_code = get_weather_code(ph_hint, fog, current['gmt_string'], cloud_cover, current['wind_velocity_hint'], sunrise, sunset)
    
    window.setProperty("Current.OutlookIcon", f'{icon_code}.png')
    window.setProperty("Current.FanartCode", f'{icon_code}')

    window.setProperty("Current.WindDirection", get_wind_direction(current['wind_direction_hint']))
    window.setProperty("Current.Wind", f"{str(current['wind_velocity']['kmh'])}")

    window.setProperty("Current.Humidity", f"{current['humidity']}")

    # Загадка для меня, но восход и закат заполняются ТОЛЬКО с Today!
    # если указать Current - НЕ ПОЯВЛЯЮТСЯ!!!
    window.setProperty("Today.Sunrise", sunrise)
    window.setProperty("Today.Sunset", sunset)

    window.setProperty("Current.Pressure", f"{current['pressure']['mmhg']} мм рт. ст.")

    #Объявляем эту часть погоды (current) заполненной и готовой к показу
    window.setProperty("Current.IsFetched", "true")


    # В цикле заполняем плашки первого(верхнего) ряда - погода на 14 дней
    for i, hours_in_day in enumerate(daily,1):
        # !!! ВАЖНО !!!
        # Питоновские циклы начинаются с 0, а элементы коди нумеруются с 1!
        # В значении часа добавляем 1 (i+1), иначе первое значение (день/час) из цикла НЕ ОТОБРАЖАЕТСЯ!
        window.setProperty(f"Daily.{i}.ShortDate", hours_in_day[0]['day'])
        window.setProperty(f"Daily.{i}.LongDate", hours_in_day[0]['day'])
        window.setProperty(f"Daily.{i}.ShortDay", hours_in_day[0]['day_week'])
        window.setProperty(f"Daily.{i}.LongDay", hours_in_day[0]['day_week'])
        
        low_temp = min(map(lambda x: x['temperature']['c'], hours_in_day))
        high_temp = max(map(lambda x: x['temperature']['c'], hours_in_day))
        avg_temp = int((lambda lst: sum(x['temperature']['c'] for x in lst) / len(lst))(hours_in_day))
        avg_humidity = int((lambda lst: sum(x['humidity'] for x in lst) / len(lst))(hours_in_day))
        avg_pressure = int((lambda lst: sum(x['pressure']['mmhg'] for x in lst) / len(lst))(hours_in_day))
        avg_prec_rain = int((lambda lst: sum(get_phenomenon(x['phenomenon'])[1] for x in lst) / len(lst))(hours_in_day))
        avg_prec_snow = int((lambda lst: sum(get_phenomenon(x['phenomenon'])[2] for x in lst) / len(lst))(hours_in_day))  
        avg_fog = int((lambda lst: sum(get_phenomenon(x['phenomenon'])[3] for x in lst) / len(lst))(hours_in_day))  
        avg_cloud_cover = int((lambda lst: sum(x['cloud_cover']['pct'] for x in lst) / len(lst))(hours_in_day))
        avg_wind_velocity = int((lambda lst: sum(x['wind_velocity_hint'] for x in lst) / len(lst))(hours_in_day))
        avg_ph_hint = max((v := [get_phenomenon(x['phenomenon'])[0] for x in hours_in_day]), key=v.count)
        
        window.setProperty(f"Daily.{i}.LowTemperature", f"{low_temp}°C")
        window.setProperty(f"Daily.{i}.HighTemperature", f"{high_temp}°C")

        sunrise = ru_format_date(hours_in_day[0]['sunrise'])[2]
        sunset = ru_format_date(hours_in_day[0]['sunset'])[2]
        icon_code = get_weather_code(avg_ph_hint, avg_fog, '14:00', avg_cloud_cover, avg_wind_velocity, sunrise, sunset)

        window.setProperty(f"Daily.{i}.OutlookIcon", f"{icon_code}.png")
        window.setProperty(f"Daily.{i}.FanartCode", f"{icon_code}")        
        
        window.setProperty(f"Daily.{i}.Humidity", f"{avg_humidity}%")
        window.setProperty(f"Daily.{i}.Precipitation", f"{avg_prec_rain or avg_prec_snow} мм")
        window.setProperty(f"Daily.{i}.Pressure", f"{avg_pressure} мм рт. ст.")

        window.setProperty(f"Day{i-1}.Title", hours_in_day[0]['day'])
        window.setProperty(f"Day{i-1}.LowTemp", f"{low_temp}°C")
        window.setProperty(f"Day{i-1}.HighTemp", f"{high_temp}°C")
        window.setProperty(f"Day{i-1}.Outlook", avg_ph_hint)
        window.setProperty(f"Day{i-1}.FanartCode", f"{icon_code}")


        # !!!!! ВАЖНО !!!!!
        # Хотя в скине Estuary значения Outlook не отображаются, НО
        # без заполнения Outlook плашки СОЗДАВАТЬСЯ НЕ БУДУТ!!!
        window.setProperty(f"Daily.{i}.Outlook", avg_ph_hint)

    # Объявляем эту часть погоды (daily) заполненной и готовой к показу
    window.setProperty("Daily.IsFetched", "true")


    # В цикле заполняем плашки второго(нижнего) ряда - погода на 24 часа
    for i, hour in enumerate(hourly,1):
        h_day, h_week, h_time = ru_format_date(hour['gmt'])
        window.setProperty(f"Hourly.{i}.Time", h_time)
        window.setProperty(f"Hourly.{i}.ShortDate", h_day)
        window.setProperty(f"Hourly.{i}.LongDate", h_day)
        window.setProperty(f"Hourly.{i}.Temperature", f"{round(hour['temperature']['c'])}°C")
        ph_hint, rain_mm, snow_mm, fog = get_phenomenon(hour['phenomenon'])
        window.setProperty(f"Hourly.{i}.Precipitation", f"{rain_mm or snow_mm} мм")
        sunrise = ru_format_date(hour['sunrise'])[2]
        sunset = ru_format_date(hour['sunset'])[2]
        icon_code = get_weather_code(ph_hint, fog, hour['gmt_string'], hour['cloud_cover']['pct'], hour['wind_velocity_hint'], sunrise, sunset)
        window.setProperty(f"Hourly.{i}.OutlookIcon", f'{icon_code}.png')
        window.setProperty(f"Hourly.{i}.FanartCode", f'{icon_code}')
        window.setProperty(f"Hourly.{i}.FeelsLike", f"{round(hour['feel_temperature']['c'])}°C")
        window.setProperty(f"Hourly.{i}.WindSpeed", f"{str(hour['wind_velocity']['kmh'])}")
        window.setProperty(f"Hourly.{i}.WindDirection", get_wind_direction(hour['wind_direction_hint']))
        window.setProperty(f"Hourly.{i}.Humidity", f"{hour['humidity']}")
        window.setProperty(f"Hourly.{i}.Pressure", f"{current['pressure']['mmhg']} мм рт. ст.")
        
        # !!!!! ВАЖНО !!!!!
        # Хотя в скине Estuary значения Outlook не отображаются, НО
        # без заполнения Outlook плашки СОЗДАВАТЬСЯ НЕ БУДУТ!!!
        window.setProperty(f"Hourly.{i}.Outlook", f"{ph_hint}") 

    # Объявляем эту часть погоды (hourly) заполненной и готовой к показу
    window.setProperty("Hourly.IsFetched", "true")
    

def get_parse_weather(geoid):
    weather_data = get_weather(geoid)
    seconds_offset = get_weather(geoid,'a')['gmt_add']
    data = filter_future_hours_only(weather_data,seconds_offset)
    # 1. current_w
    current_w = data[0]

    # 2. daily_w
    daily_w = {}
    day_data = [{**x, 'day': ru_format_date(x['gmt'])[0], 'day_week': ru_format_date(x['gmt'])[1]} for x in data if  (7 < int(x['gmt_string'][:2]) < 16)]
    for x in day_data:
        old_value = daily_w.setdefault(x['day'],[])
        old_value.append(x)
        daily_w[x['day']] = old_value

    # 3. hourly_w
    hourly_w = data[1:13]

    return current_w, daily_w.values(), hourly_w


def ru_format_date(ts):
    days = ["Понедельник","Вторник","Среда","Четверг","Пятница","Суббота","Воскресенье"]
    months = ["января","февраля","марта","апреля","мая","июня","июля","августа","сентября","октября","ноября","декабря"]
    dt = datetime.utcfromtimestamp(ts).timetuple()
    return f"{dt.tm_mday} {months[dt.tm_mon-1]}", days[dt.tm_wday], f"{dt.tm_hour:02}:{dt.tm_min:02}"


# Функция для показа только тех часов погоды, которые больше текущего времени
def filter_future_hours_only(weather_data,seconds_offset):
    gmt_time = int(datetime.now().timestamp()) + seconds_offset - 60*60
    weather_data = [x for x in weather_data if x['gmt']>=gmt_time]
    return sorted(weather_data, key=lambda x: x['gmt'])



def determine_location ():
    # Получаем все заполненные локации из settings.xml
    locs = []
    for i in range(1, 4):
        loc_value = ADDON.getSetting(f'loc{i}_id')  # Получаем значение
        if loc_value and loc_value.strip():  # Проверяем не пустое ли значение
            locs.append(loc_value.strip())  # Добавляем без пробелов по краям

    if not locs:
        locs = handle_location_search()
    # Сообщаем Коди сколько всего локаций ввел пользователь
    # количество локаций вводится как СТРОКА, НЕ как число!!!
    window.setProperty("Locations", str(len(locs)))

    # Заполняем Location№ введенными пользователем названиями для отображения в погоде
    for i in range(1, 4):
        location_name = ADDON.getSetting('loc{}_name'.format(i)).strip()
        if location_name:  # Если хотя бы одно имя есть
            window.setProperty("Location{}".format(i), location_name)


# Использовалась при отладке для вывода сообщений
def gui_message (title, message):
    dialog = xbmcgui.Dialog()
    dialog.ok(title, message)
