import requests
import xbmc
import xbmcgui
import xbmcaddon
import json
from consts import (
    RU_DAYS,
    RU_MONTHS,
    RU_WIND_DIR,
    CONDITIONS
)


window = xbmcgui.Window(12600)
ADDON = xbmcaddon.Addon()

sunrise_setting = ADDON.getSetting('sunrise')
sunset_setting = ADDON.getSetting('sunset')
precipitation_setting = ADDON.getSetting('precipitation')
sunrise_sunset_options = ['Начало и конец','Начало','Конец']
precipitation_options = ['В мм','Вероятность']
sunrise_id = sunrise_sunset_options.index(sunrise_setting)
sunset_id = sunrise_sunset_options.index(sunset_setting)
precipitation_id = precipitation_options.index(precipitation_setting)



class LocationSearch:
    def __init__(self):
        self.dialog = xbmcgui.Dialog()
        self.keyboard = xbmc.Keyboard()
    
    def search_location(self):
        self.keyboard.setHeading("Введите название города")
        self.keyboard.doModal()        
        if self.keyboard.isConfirmed():
            query = self.keyboard.getText()
            if query:
                return self._perform_search(query)
        return None
    
    def _perform_search(self, query):
        locations = self._search_weather_api(query)      
        if not locations:
            self.dialog.ok("Я.Погода", "Локации не найдены") 
            return None
        return self._show_location_list(locations)
    
    def _search_weather_api(self, query):        
        url = f"https://suggest-maps.yandex.ru/suggest-geo?v=7&n=5&callback=&part={query}"
        return requests.get(url).json()['results']
    
    def _show_location_list(self, locations):
        items = [f"{location['name']} [{location.get('desc','')}]" for location in locations]        
        selected = self.dialog.select("Выберите локацию", items)
        if selected >= 0:
            return locations[selected]
        return None


def handle_location_search(loc_id=1):
    search = LocationSearch()
    location = search.search_location()    
    if location:
        # Сохраняем выбранную локацию в настройках
        ADDON.setSetting(f"loc{loc_id}_id", str(location["geoid"]))
        ADDON.setSetting(f"loc{loc_id}_name", location["name"])        
        # Показываем подтверждение
        xbmcgui.Dialog().ok("Я.Погода", f"Местоположение установлено: {location['name']}")
        return list(location)
    return []


def set_weather (current: dict, daily: list, hourly: list, args: str):
    # Заполняем шапку
    location_value = window.getProperty(f"Location{args}")
    window.setProperty("Current.Location", location_value)
    window.setProperty("WeatherProvider", "Я.Погода")

    window.setProperty("Current.Temperature", f"{round(current['temp_c'])}°C")
    window.setProperty("Current.FeelsLike", f"{round(current['feelslike_c'])}°C")
    window.setProperty("Current.Condition", current['condition']['text'])

    # В шапке этого значка не будет, но он появится под часами, если включить опцию
    # "Интерфейс - Настройки обложки - Показать сведения о погоде на верхней панели"
    window.setProperty("Current.OutlookIcon", current['condition']['code'])
    window.setProperty("Current.FanartCode", f"{current['condition']['code'][:-4]}")

    window.setProperty("Current.WindDirection", current['wind_dir'])
    window.setProperty("Current.Wind", f"{str(current['wind_kph'])}")

    window.setProperty("Current.Humidity", f"{current['humidity']}")
    window.setProperty("Current.Precipitation", f"{current['prec_info']}")

    # Загадка для меня, но восход и закат заполняются ТОЛЬКО с Today!
    # если указать Current - НЕ ПОЯВЛЯЮТСЯ!!!
    window.setProperty("Today.Sunrise", f"{current['vosxod']}")
    window.setProperty("Today.Sunset", f"{current['zakat']}")

    window.setProperty("Current.Cloudiness", str(current['cloudness']*100))
    window.setProperty("Current.UVIndex", str(current['uv_index']))
    window.setProperty("Current.Pressure", str(current['pressure']))
    window.setProperty("Current.Visibility", str(current['visibility']))

    #Объявляем эту часть погоды (current) заполненной и готовой к показу
    window.setProperty("Current.IsFetched", "true")


    # В цикле заполняем плашки первого(верхнего) ряда - погода на 14 дней
    for i, day in enumerate(daily,1):

        # !!! ВАЖНО !!!
        # Питоновские циклы начинаются с 0, а элементы коди нумеруются с 1!
        # В значении часа добавляем 1 (i+1), иначе первое значение (день/час) из цикла НЕ ОТОБРАЖАЕТСЯ!
        window.setProperty(f"Daily.{i}.ShortDate", day['date'])
        window.setProperty(f"Daily.{i}.LongDate", day['date'])
        window.setProperty(f"Daily.{i}.ShortDay", day['day_of_week'])
        window.setProperty(f"Daily.{i}.LongDay", day['day_of_week'])
        window.setProperty(f"Daily.{i}.LowTemperature", f"{round(day['temp_min'])}°C")
        window.setProperty(f"Daily.{i}.HighTemperature", f"{round(day['temp'])}°C")
        window.setProperty(f"Daily.{i}.OutlookIcon", day['condition']['code'])
        window.setProperty(f"Daily.{i}.FanartCode", f"{day['condition']['code'][:-4]}")
        window.setProperty(f"Daily.{i}.Humidity", f"{day['humidity']}%")
        window.setProperty(f"Daily.{i}.Precipitation", [f"{day['prec_mm']} мм",f"{day['prec_prob']}%"][precipitation_id])
        window.setProperty(f"Daily.{i}.Pressure", str(day['pressure_mm']))


        window.setProperty(f"Day{i-1}.Title", day['date'])
        window.setProperty(f"Day{i-1}.LowTemp", f"{round(day['temp_min'])}°C")
        window.setProperty(f"Day{i-1}.HighTemp", f"{round(day['temp'])}°C")
        window.setProperty(f"Day{i-1}.Outlook", day['condition']['text'])
        window.setProperty(f"Day{i-1}.FanartCode", day['condition']['code'][:-4])


        # !!!!! ВАЖНО !!!!!
        # Хотя в скине Estuary значения Outlook не отображаются, НО
        # без заполнения Outlook плашки СОЗДАВАТЬСЯ НЕ БУДУТ!!!
        window.setProperty(f"Daily.{i}.Outlook", f"{day['condition']['text']}")

    # Объявляем эту часть погоды (daily) заполненной и готовой к показу
    window.setProperty("Daily.IsFetched", "true")


    # В цикле заполняем плашки второго(нижнего) ряда - погода на 24 часа
    for i, hour in enumerate(hourly,1):

        # !!! ВАЖНО !!!
        # Питоновские циклы начинаются с 0, а элементы коди нумеруются с 1!
        # Поэтому в значении часа добавляем 1 (i+1)
        window.setProperty(f"Hourly.{i}.Time", f"{hour['time']}")
        window.setProperty(f"Hourly.{i}.ShortDate", f"{hour['date']}")
        window.setProperty(f"Hourly.{i}.LongDate", f"{hour['date']}")
        window.setProperty(f"Hourly.{i}.Temperature", f"{round(hour['temp'])}°C")
        window.setProperty(f"Hourly.{i}.Precipitation", [f"{hour['prec_mm']} мм",f"{hour['prec_prob']}%"][precipitation_id])
        window.setProperty(f"Hourly.{i}.OutlookIcon", hour['condition']['code'])
        window.setProperty(f"Hourly.{i}.FanartCode", f"{hour['condition']['code'][:-4]}")
        window.setProperty(f"Hourly.{i}.FeelsLike", f"{hour['feels_like']}")
        window.setProperty(f"Hourly.{i}.WindSpeed", f"{hour['wind_dir']}")
        window.setProperty(f"Hourly.{i}.WindDirection", RU_WIND_DIR.get(hour["wind_dir"], hour["wind_dir"]))
        window.setProperty(f"Hourly.{i}.Humidity", f"{hour['humidity']}%")
        window.setProperty(f"Hourly.{i}.UVIndex", f"{hour['uv_index']}")
        window.setProperty(f"Hourly.{i}.Outlook", hour['condition']['code'])

        # !!!!! ВАЖНО !!!!!
        # Хотя в скине Estuary значения Outlook не отображаются, НО
        # без заполнения Outlook плашки СОЗДАВАТЬСЯ НЕ БУДУТ!!!
        window.setProperty(f"Hourly.{i}.Outlook", hour['condition']['text'])

    # Объявляем эту часть погоды (hourly) заполненной и готовой к показу
    window.setProperty("Hourly.IsFetched", "true")


def yandex_icon_to_kodi(icon_code, cloudiness=0):
    mapping = {
        # Ясная погода
        'skc_d': 32,  # sunny
        'skc_n': 31,  # clear (night)        
        # Туман
        'fg_d': 20, 'fg_n': 20,  # foggy
        # Переменная облачность
        'bkn_d': 30,  # partly cloudy (day)
        'bkn_n': 29,  # partly cloudy (night)
        # Слабый дождь при переменной облачности
        'bkn_-ra_d': 9, 'bkn_-ra_n': 9,  # drizzle
        'bkn_-sn_d': 14, 'bkn_-sn_n': 14,  # light snow showers
        # Дождь/снег при переменной облачности
        'bkn_ra_d': 11, 'bkn_ra_n': 11,  # showers
        'bkn_sn_d': 16, 'bkn_sn_n': 16,  # snow
        # Сильный дождь/снег при переменной облачности
        'bkn_+ra_d': 12, 'bkn_+ra_n': 12,  # showers
        'bkn_+sn_d': 41, 'bkn_+sn_n': 41,  # heavy snow
        # Грозы
        'ovc_ts': 4,  # thunderstorms
        'ovc_ts_ra': 4,  # thunderstorms
        'ovc_ts_ha': 35,  # mixed rain and hail
        # Пасмурно
        'ovc': 26,  # cloudy
        # Пасмурно со слабыми осадками
        'ovc_-ra': 9,  # drizzle
        'ovc_-sn': 14,  # light snow showers
        # Пасмурно с осадками
        'ovc_ra': 11,  # showers
        'ovc_sn': 16,  # snow
        # Пасмурно с сильными осадками
        'ovc_+ra': 12,  # showers
        'ovc_+sn': 41,  # heavy snow
        # Смешанные осадки
        'ovc_ra_sn': 5,  # mixed rain and snow
        # Град
        'ovc_ha': 17,  # hail
        # Метели
        '-bl': 15,  # blowing snow
        'bl': 15,  # blowing snow
        # Пыль и смог
        'dst': 19,  # dust
        'du_st': 19,  # dust
        'smog': 22,  # smoky
        # Экстремальные явления
        'strm': 2,  # hurricane
        'vlka': 22,  # smoky
    }
    
    # Получаем базовый код
    kodi_code = mapping.get(icon_code, 'na')
    
    # Дополнительная логика для уточнения облачности, если доступен процент
    if cloudiness is not None and 0 <= cloudiness <= 1:
        # Для ясной погоды с небольшой облачностью - используем "fair"
        if icon_code in ['skc_d', 'skc_n'] and cloudiness > 0.1:
            if 'd' in icon_code:
                kodi_code = 34  # fair (day)
            else:
                kodi_code = 33  # fair (night)
        # Для переменной облачности уточняем степень облачности
        elif icon_code in ['bkn_d', 'bkn_n']:
            if cloudiness > 0.7:
                # Сильная облачность, но не сплошная
                kodi_code = 28 if 'd' in icon_code else 27  # mostly cloudy
            elif cloudiness < 0.3:
                # Слабая облачность
                kodi_code = 34 if 'd' in icon_code else 33  # fair    
    return kodi_code




def update_condition(data):
    # Если это один элемент (не список), оборачиваем его в список
    if not isinstance(data, list):
        data = [data]
    for item in data:
        icon = yandex_icon_to_kodi(item["icon"],item["cloudness"])
        item["condition"] = {
            "text": CONDITIONS[item["condition"]].capitalize(),
            "code": f"{icon}.png"
        }


def parse_weather(data: dict):
    # 1. current_w
    current = data["fact"]
    wind_dir_ru = RU_WIND_DIR.get(current["wind_dir"], current["wind_dir"])
    astro = data["forecasts"][0]
    vosxod = [f"{astro.get('rise_begin')}-{astro.get('sunrise')}",f"{astro.get('rise_begin')}",f"{astro.get('sunrise')}"][sunrise_id]
    zakat = [f"{astro.get('sunset')}-{astro.get('set_end')}",f"{astro.get('sunset')}",f"{astro.get('set_end')}"][sunset_id]


    current_w = {
        "temp_c": current["temp"],
        "is_day": 1 if 'd'==current["daytime"] else 0,
        "condition": current["condition"],
        "wind_kph": current["wind_speed"]*3.6, # перевод в км/ч
        "wind_dir": wind_dir_ru,
        "prec_info": [f"{current.get('prec_mm',0)} мм",f"{current.get('prec_prob',0)}%"][precipitation_id],
        "humidity": current["humidity"],
        "feelslike_c": current["feels_like"],
        "vosxod": vosxod,
        "zakat": zakat,
        "icon": current["icon"],
        "cloudness": current["cloudness"],
        "pressure": current["pressure_mm"],
        "visibility": current["visibility"],
        "uv_index": current["uv_index"],
    }

    update_condition(current_w)

    # 2. daily_w
    daily_w = []

    for forecast_day in data["forecasts"]:
        # Получаем день недели
        day_of_week = get_day_of_week(forecast_day["date"])
        ru_date = ru_format_date(forecast_day["date"])
        # Формируем запись
        day_data = {
            "date": ru_date,
            "day_of_week": day_of_week,  # Добавляем день недели
            **forecast_day["parts"]["day_short"]  # Остальные поля (maxtemp_c, mintemp_c...)
        }
        daily_w.append(day_data)

    update_condition(daily_w)


    # 3. hourly_w
    hourly_w = []

    for day in data['forecasts']:
        for hour in day['hours']:
            ru_date = ru_format_date(day['date'])
            # Формируем новый словарь с правильной структурой
            hour_data = {
                'date': ru_date,  # '2025-05-02' (взято из time)
                'time': f"{hour['hour']}:00",
                **hour  # Остальные поля (temp_c, condition и т.д.)
            }
            hourly_w.append(hour_data)

    update_condition(hourly_w)

    return current_w, daily_w, hourly_w



def get_weather(slug):
    """Получить данные о погоде с Yandex Weather"""
    url = f'https://api.weather.yandex.ru/v1/forecast?l10n=false&slug={slug}&lang=ru_RU'
    headers = {
        'X-Yandex-Weather-Client': 'YandexWeatherAndroid/5.2',
        'X-Yandex-Weather-Token': 'f7d3d928330f0d0347c39afc04fe2ded',
        'X-Yandex-Weather-Timestamp': '1748774978',
    }
    response = requests.get(url, headers=headers)
    data = response.json()
    filter_future_hours_only(data)
    return data



def ru_format_date(date_str):
    year, month, day = map(int, date_str.split('-'))

    return f"{day} {RU_MONTHS[month]}"



def get_day_of_week(date_str):
    # Используем формулу Зеллера для получения дня недели из даты
    year, month, day = map(int, date_str.split('-'))

    if month < 3:
        month += 12
        year -= 1

    q = day         # День месяца
    m = month       # Месяц (от 3 до 14)
    K = year % 100  # Последние две цифры года
    J = year // 100 # Первые две цифры года

    h = (q + ((13 * (m + 1)) // 5) + K + (K // 4) + (J // 4) - (2 * J)) % 7
    # Возвращаем 0 = Saturday, 1 = Sunday, ..., 6 = Friday

    return RU_DAYS[str(h)]


def convert_time(time_str):
    """
    Конвертирует время из формата 'hh:mm AM/PM' в 24-часовой формат 'H:mm'.
    Возвращает часы без ведущего нуля, но минуты — всегда с нулём.

    :param time_str: Время в формате 'hh:mm AM/PM' (например, '06:21 AM').
    :return: Время в 24-часовом формате (например, '6:21').
    """
    time_part, am_pm = time_str.split()
    hours, minutes = map(int, time_part.split(':'))

    if am_pm.upper() == "PM" and hours != 12:
        hours += 12
    elif am_pm.upper() == "AM" and hours == 12:
        hours = 0

    return f"{hours}:{minutes:02d}"  # Часы как есть, минуты с нулём


# Функция для показа только тех часов погоды, которые больше текущего времени
def filter_future_hours_only(weather_data):
    current_timestamp = weather_data["now"]
    for day in weather_data["forecasts"]:
        if "hours" in day:
            # Оставляем только те часы, которые > current_time_str
            day["hours"] = [h for h in day["hours"] if h["hour_ts"] > current_timestamp]


def determine_location ():
    # Получаем все заполненные локации из settings.xml
    locs = []
    for i in range(1, 4):
        loc_value = ADDON.getSetting(f'loc{i}_id')  # Получаем значение
        if loc_value and loc_value.strip():  # Проверяем не пустое ли значение
            locs.append(loc_value.strip())  # Добавляем без пробелов по краям

    if not locs:
        locs = handle_location_search()
    # Сообщаем Коди сколько всего локаций ввел пользователь
    # количество локаций вводится как СТРОКА, НЕ как число!!!
    window.setProperty("Locations", str(len(locs)))

    # Заполняем Location№ введенными пользователем названиями для отображения в погоде
    for i in range(1, 4):
        location_name = ADDON.getSetting('loc{}_name'.format(i)).strip()
        if location_name:  # Если хотя бы одно имя есть
            window.setProperty("Location{}".format(i), location_name)

# Использовалась при отладке для вывода сообщений
def gui_message (title, message):
    dialog = xbmcgui.Dialog()
    dialog.ok(title, message)
